#include "CFile.h"

// constructor
CFile::CFile()
{
FileDescriptor=-1;
}

// constructor
CFile::CFile(CString FileName, int OpenFlags, mode_t Permissions)
{
FileDescriptor=-1;
Open(FileName, OpenFlags, Permissions);
}

// constructor
CFile::CFile(int srcFileDescriptor)
{
if (srcFileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalParameter, "CFile::CFile"));

FileDescriptor=-1;
Attach(srcFileDescriptor);
}

CFile::CFile(CFile& srcFile)
{
if (srcFile.FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalParameter, "CFile::CFile"));

// copy handle
FileDescriptor=srcFile.FileDescriptor;
}

// destructor
CFile::~CFile()
{
if (FileDescriptor != -1) Close();
}

// static methods
mode_t CFile::SetFileCreateMask(mode_t newCreateMask)
{
return umask(newCreateMask);
}

// static methods
void CFile::Delete(CString FileName)
{
if (unlink((const char *)FileName) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Delete"));
}

void CFile::GetStatus(CString FileName, struct stat *destBuffer)
{
if (stat((const char *)FileName, destBuffer) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetStatus"));
}

void CFile::SetPermission(CString FileName, mode_t newPermissions)
{
if (chmod((const char *)FileName, newPermissions) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetPermission"));
}

void CFile::SetOwnerUid(CString FileName, uid_t srcOwnerUid)
{
if (chown((const char *)FileName, srcOwnerUid, (gid_t)(-1)) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetOwnerUid"));
}

void CFile::SetOwnerGid(CString FileName, gid_t srcOwnerGid)
{
if (chown((const char *)FileName, (uid_t)(-1), srcOwnerGid) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetOwnerGid"));
}

// static methods
int CFile::Wait(CArray<CFile *, CFile *>& ReadFileArray, CArray<CFile *, CFile *>& WriteFileArray,
	CArray<CFile *, CFile *>& ExceptionFileArray, int WaitSec, int WaitMicroSec)
{
fd_set readfd, writefd, exceptionfd;
int i, maxfd, ret;
struct timeval tm;

maxfd=0;
FD_ZERO(&readfd);
FD_ZERO(&writefd);
FD_ZERO(&exceptionfd);
for(i=0;i<ReadFileArray.Size;i++)
	{
	FD_SET((ReadFileArray[i])->FileDescriptor, &readfd);
	maxfd=max(maxfd, (ReadFileArray[i])->FileDescriptor);
	}
for(i=0;i<WriteFileArray.Size;i++)
	{
	FD_SET((WriteFileArray[i])->FileDescriptor, &writefd);
	maxfd=max(maxfd, (WriteFileArray[i])->FileDescriptor);
	}
for(i=0;i<ExceptionFileArray.Size;i++)
	{
	FD_SET((ExceptionFileArray[i])->FileDescriptor, &exceptionfd);
	maxfd=max(maxfd, (ExceptionFileArray[i])->FileDescriptor);
	}
if (WaitSec >= 0)
	{
	tm.tv_sec=WaitSec;
	tm.tv_usec=WaitMicroSec;
	ret=select(maxfd+1, &readfd, &writefd, &exceptionfd, &tm);
	}
else	{
	ret=select(maxfd+1, &readfd, &writefd, &exceptionfd, NULL);
	}
return ret;
}

// static methods
void CFile::SyncAll()
{
sync();
}

// methods
void CFile::Attach(int srcFileDescriptor)
{
if (FileDescriptor != -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Attach"));
FileDescriptor=srcFileDescriptor;
}

// methods
int CFile::Detach()
{
int destFileDescriptor;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Detach"));
destFileDescriptor=FileDescriptor;
FileDescriptor=-1;
return destFileDescriptor;
}

// methods
void CFile::Open(CString FileName, int OpenFlags, mode_t Permissions)
{
if (FileDescriptor != -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Attach"));

FileDescriptor=open((const char *)FileName, OpenFlags, Permissions);
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Open"));	
}

// methods
void CFile::Close()
{
if (FileDescriptor != -1)
	{
	if (close(FileDescriptor) == -1)
		throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Open"));
	FileDescriptor=-1;
	}
}

// methods
int CFile::ReadByte()
{
char unsigned buffer[1];
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::ReadByte"));

ret=read(FileDescriptor, (void *)buffer, 1);
if (ret > 0)
	return (int)buffer[1];
else if (ret < 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::ReadByte"));
else	return -1;
}

// methods
int CFile::WriteByte(char unsigned Data)
{
char unsigned buffer[1];
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::WriteByte"));

buffer[0]=Data;
ret=write(FileDescriptor, (void *)buffer, 1);
if (ret > 0)
	return 1;
else if (ret < 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::WriteByte"));
else	return 0;
}

// methods
int CFile::Read(void *Buffer, size_t BufferSize)
{
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Read"));

ret=read(FileDescriptor, (void *)Buffer, BufferSize);
if (ret < 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Read"));
else	return ret;
}

// methods
int CFile::Write(void *Buffer, size_t BufferSize)
{
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Write"));

ret=write(FileDescriptor, (void *)Buffer, BufferSize);
if (ret < 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Write"));
else	return ret;
}

// methods
void CFile::Sync()
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Sync"));

if (fsync(FileDescriptor) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Sync"));
}

// methods
off_t CFile::Seek(int Whence, off_t Position)
{
off_t ret;

ret=lseek(FileDescriptor, Position, Whence);
if (ret == (off_t)(-1))
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Seek"));
return ret;
}

// methods
void CFile::Rewind()
{
if (lseek(FileDescriptor, (off_t)0, SEEK_SET) == (off_t)(-1))
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Seek"));
}

// methods
CFile& CFile::Duplicate()
{
CFile *newFile;
int newFileDescriptor;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Duplicate"));

// duplicate handle
newFileDescriptor=dup(FileDescriptor);
if (newFileDescriptor == -1)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::Duplicate"));

newFile=new CFile();
newFile->Attach(newFileDescriptor);
return (*newFile);
}

// methods
CFile& CFile::Copy()
{
CFile *newFile;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::Copy"));

// copy handle
newFile=new CFile();
newFile->Attach(FileDescriptor);
return (*newFile);
}

// methods
int CFile::GetCloseOnExecFlag()
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetCloseOnExecFlag"));

return fcntl(FileDescriptor, F_GETFD);
}

// methods
void CFile::SetCloseOnExecFlag(int CloseOnExecFlag)
{
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetCloseOnExecFlag"));

if (fcntl(FileDescriptor, F_SETFD, CloseOnExecFlag) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetCloseOnExecFlag"));
}

// methods
int CFile::GetOpenFlags()
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetOpenFlags"));

return fcntl(FileDescriptor, F_GETFL);
}

// methods
void CFile::SetOpenFlags(int OpenFlags)
{
int ret;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetOpenFlags"));

if (fcntl(FileDescriptor, F_SETFL, OpenFlags) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetOpenFlags"));
}

// methods
int CFile::GetLock(int LockFlag, int Whence, off_t StartPos, off_t Length,
	int *lockedWhence, off_t *lockedStartPos, off_t *lockedLength, int *lockedPid)
{
struct flock fl;

fl.l_type=LockFlag;
fl.l_whence=Whence;
fl.l_start=StartPos;
fl.l_len=Length;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetLock"));

if (fcntl(FileDescriptor, F_GETLK, &fl) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetLock"));

if (fl.l_type == F_UNLCK) return 0;
else	{
	if (lockedWhence != NULL) (*lockedWhence)=fl.l_whence;
	if (lockedStartPos != NULL) (*lockedStartPos)=fl.l_start;
	if (lockedLength != NULL) (*lockedLength)=fl.l_len;
	if (lockedPid != NULL) (*lockedPid)=fl.l_pid;

	return 1;
	}
}

void CFile::SetLockAlarmSignalProc(int Signal)
{
// do nothing
return;
}

// methods
void CFile::SetLock(int LockFlag, int Whence, off_t StartPos, off_t Length, int TimeoutSec)
{
struct flock fl;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetLock"));

fl.l_type=LockFlag;
fl.l_whence=Whence;
fl.l_start=StartPos;
fl.l_len=Length;

if (TimeoutSec < 1)
	{
	// return immediately
	if (fcntl(FileDescriptor, F_SETLK, &fl) < 0)
		throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetLock"));
	}
else	{
	struct sigaction act, oldact;
	sigset_t sigmask;
	int ret;

	// set signalset (fill)
	sigemptyset(&sigmask);
	sigfillset(&sigmask);

	// set alarm
	memset((void *)(&act), 0, sizeof(struct sigaction));
	act.sa_handler=CFile::SetLockAlarmSignalProc;
	act.sa_mask=sigmask;	// all of signals are masked
	act.sa_flags=0;
	memset((void *)(&oldact), 0, sizeof(struct sigaction));

	if (sigaction(SIGALRM, &act, &oldact) != 0)
		throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetLock"));

	alarm(TimeoutSec);

	// return if successful, or if alarm
	ret=fcntl(FileDescriptor, F_SETLKW, &fl);

	// cancel alarm proc
	if (sigaction(SIGALRM, &oldact, &act) != 0)
		throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetLock"));

	if (ret < 0)
		throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetLock"));
	}
}

// methods
void CFile::GetStatus(struct stat *destBuffer)
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetStatus"));

if (fstat(FileDescriptor, destBuffer) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetStatus"));
}

// methods
void CFile::SetPermission(mode_t newPermissions)
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetPermission"));

if (fchmod(FileDescriptor, newPermissions) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetPermission"));
}

// methods
void CFile::SetOwnerUid(uid_t srcOwnerUid)
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetOwnerUid"));

if (fchown(FileDescriptor, srcOwnerUid, (gid_t)(-1)) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetOwnerUid"));
}

// methods
void CFile::SetOwnerGid(gid_t srcOwnerGid)
{
if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::SetOwnerGid"));

if (fchown(FileDescriptor, (uid_t)(-1), srcOwnerGid) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::SetOwnerGid"));
}

// methods
off_t CFile::GetFileSize()
{
struct stat buf;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetFileSize"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetFileSize"));
return buf.st_size;
}

// methods
CTime CFile::GetLastAccessTime()
{
struct stat buf;
CTime LastAccessTime;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetLastAccessTime"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetLastAccessTime"));
LastAccessTime=(const time_t)buf.st_atime;

return LastAccessTime;
}

// methods
CTime CFile::GetLastModifiedTime()
{
struct stat buf;
CTime LastModifiedTime;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetLastModifiedTime"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetLastModifiedTime"));
LastModifiedTime=(const time_t)buf.st_atime;

return LastModifiedTime;
}

// methods
CTime CFile::GetLastChangedTime()
{
struct stat buf;
CTime LastChangedTime;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetLastChangedTime"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetLastChangedTime"));
LastChangedTime=(const time_t)buf.st_atime;

return LastChangedTime;
}

// methods
uid_t CFile::GetOwnerUid()
{
struct stat buf;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetOwnerUid"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetOwnerUid"));
return buf.st_uid;
}

// methods
gid_t CFile::GetOwnerGid()
{
struct stat buf;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetOwnerGid"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetOwnerGid"));
return buf.st_gid;
}

// methods
mode_t CFile::GetPermission()
{
struct stat buf;

if (FileDescriptor == -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::GetPermission"));

if (fstat(FileDescriptor, &buf) != 0)
	throw(CException(CException::CErrorTypeErrorCode, errno, "CFile::GetPermission"));
return buf.st_mode;
}

// methods
CString CFile::ReadLine(int MaxLength)
{
/* CString Line;
int ch;

Line="";
while ((ch=ReadByte()) != -1)
	{
	if ((ch == 13)||(ch == 10))
		{
		// CR+LF̉sR[hȂ̂
		Line+=(char unsigned)ch;	// sR[h
		if (ch == 13)
			{
			ch=ReadByte();
			if (ch != 10) ungetc(ch, hFile); // LFł͂Ȃ̂Ŗ߂
			else 	Line+=(char unsigned)ch; // sR[h
			}
		break;
		}

	Line+=(char unsigned)ch;
	if (Line.Length >= MaxLength) break;
	}

return Line; */
}

// operators
CFile& CFile::operator=(const CFile& srcFile)
{
if (FileDescriptor != -1)
	throw(CException(CException::CErrorTypeException, CException::CIllegalCallOrder, "CFile::operator="));

// copy handle
FileDescriptor=srcFile.FileDescriptor;

return (*this);
}

#ifdef CCLASSDEBUG
void CFile::Dump()
{
printf("FileDescriptor=%i\n", FileDescriptor);
}

void CFile::SelfTest()
{
CFile File;

printf("Self testing CFile() class..\n");

/* printf("  step 1: static methods test..\n");
printf("    SetFileCreateMask(0066) .. "); CFile::SetFileCreateMask(0066); printf("done.\n");
printf("    creating file /tmp/CFileSelfTest.bin ..");
	File.Open("/tmp/CFileSelfTest.bin", CFile::OpenFlags_Create, 0777);
	File.Close(); printf(" done.\n"); */

printf("Self testing CFile() completed.\n");
}
#endif
