#include "CFifoBuffer.h"

// constructors, destructors
CFifoBuffer::CFifoBuffer(int srcBufferSize, int srcAutoExtend, int srcAutoExtendSize)
{
BufferSize=0;
AutoExtend=srcAutoExtend;
AutoExtendSize=srcAutoExtendSize;
ReaderCursorPos=0;
WriterCursorPos=0;
Buffer=NULL;

SetBufferSize(srcBufferSize);
}

// constructors, destructors
CFifoBuffer::~CFifoBuffer()
{
if (Buffer != NULL) free(Buffer);
}

// generic methods
void CFifoBuffer::SetBufferSize(int srcnewBufferSize)
{
char unsigned *newBuffer;
int newWriterCursorPos;
int minBufferSize, newBufferSize;

// minimum buffersize
minBufferSize=GetReadableBufferSize();

if (srcnewBufferSize < 0)
	throw(CException(CException::CErrorTypeException, CException::CIllegalParameter, "CFifoBuffer::SetBufferSize"));
else	if (srcnewBufferSize == 0) newBufferSize=minBufferSize+1;	// minimum size+1 byte; read cursor & write cursor must not be overlapped
else	newBufferSize=srcnewBufferSize;

// check if new buffersize is too small
if (minBufferSize > newBufferSize)
	throw(CException(CException::CErrorTypeException, CException::CIllegalParameter, "CFifoBuffer::SetBufferSize"));

newBuffer=(char unsigned *)malloc(newBufferSize+1); // 1byte add; writer cursor must not be overlap.
if (newBuffer == NULL)
	throw(CException(CException::CErrorTypeException, errno, "CFifoBuffer::SetBufferSize"));
if (ReaderCursorPos <= WriterCursorPos)
	{
	// copy to newer allocated buffer
	memcpy((void *)newBuffer, (void *)(&(Buffer[ReaderCursorPos])), (size_t)(WriterCursorPos-ReaderCursorPos));
	newWriterCursorPos=WriterCursorPos-ReaderCursorPos;
	}
else	{
	// copy to newer allocated buffer
	memcpy((void *)newBuffer, (void *)(&(Buffer[ReaderCursorPos])), (size_t)(BufferSize-ReaderCursorPos));
	memcpy((void *)(&(newBuffer[BufferSize-ReaderCursorPos])), (void *)(Buffer), (size_t)(WriterCursorPos));
	newWriterCursorPos=(BufferSize-ReaderCursorPos)+WriterCursorPos;
	}

if (Buffer != NULL) free(Buffer);
Buffer=newBuffer;
BufferSize=newBufferSize;
ReaderCursorPos=0;
WriterCursorPos=newWriterCursorPos;
}

// generic methods
void CFifoBuffer::SetAutoExtend(int srcAutoExtend, int srcAutoExtendSize)
{
AutoExtend=srcAutoExtend;
AutoExtendSize=srcAutoExtendSize;
}

// generic methods
void CFifoBuffer::Empty()
{
ReaderCursorPos=0;
WriterCursorPos=0;
}

// reader methods
int CFifoBuffer::ReadByte()
{
char unsigned Data;

if (Read(&Data, 1) > 0)
	return Data;
else	return -1;
}

// reader methods
int CFifoBuffer::Read(char unsigned *destByteDataArray, int destByteDataArraySize)
{
char unsigned Data;
int i, ReadableSize, ReadSize;

ReadableSize=GetReadableBufferSize();
if (ReadableSize < 1) return 0;	// no data

ReadSize=min(destByteDataArraySize, ReadableSize);
for(i=0;i<ReadSize;i++)
	{
	destByteDataArray[i]=Buffer[ReaderCursorPos];
	ReaderCursorPos++; if (ReaderCursorPos >= BufferSize) ReaderCursorPos=(ReaderCursorPos%BufferSize);
	}
return ReadSize;
}

// reader methods
int CFifoBuffer::Skip(int srcBytes)
{
char unsigned Data;
int i, ReadableSize, SkipSize;

ReadableSize=GetReadableBufferSize();
if (ReadableSize < 1) return -1;	// no data

SkipSize=min(srcBytes, ReadableSize);
for(i=0;i<SkipSize;i++)
	{
	ReaderCursorPos++; if (ReaderCursorPos >= BufferSize) ReaderCursorPos=(ReaderCursorPos%BufferSize);
	}
return SkipSize;
}

// reader methods
int CFifoBuffer::GetReadableBufferSize()
{
return (WriterCursorPos >= ReaderCursorPos)?
	(WriterCursorPos-ReaderCursorPos):
	((BufferSize-ReaderCursorPos)+WriterCursorPos);
}

// writer methods
int CFifoBuffer::WriteByte(char unsigned srcByteData)
{
char unsigned copysrcByteData;

copysrcByteData=srcByteData;
return Write(&copysrcByteData, 1);
}

// writer methods
int CFifoBuffer::Write(char unsigned *srcByteDataArray, int srcArraySize)
{
int i, WriteSize;

// extend buffer if allowed
if ((GetWritableBufferSize() < srcArraySize)&&
	((AutoExtend == 1)&&(AutoExtendSize > 0)))
	{
	SetBufferSize(BufferSize+(((srcArraySize-GetWritableBufferSize())/AutoExtendSize)*AutoExtendSize)
			+((((srcArraySize-GetWritableBufferSize())%AutoExtendSize) > 0)?AutoExtendSize:0));
	}

WriteSize=min(srcArraySize, GetWritableBufferSize());
for(i=0;i<WriteSize;i++)
	{
	Buffer[WriterCursorPos]=srcByteDataArray[i];
	WriterCursorPos++; if (WriterCursorPos >= BufferSize) WriterCursorPos=(WriterCursorPos%BufferSize);
	}
return WriteSize;
}

// writer methods
int CFifoBuffer::GetWritableBufferSize()
{
return (WriterCursorPos >= ReaderCursorPos)?
	(BufferSize-(WriterCursorPos-ReaderCursorPos)-1):
	(ReaderCursorPos-WriterCursorPos-1);
}

int CFifoBuffer::PushWriteByte(char unsigned srcByteData)
{
char unsigned copysrcByteData;

copysrcByteData=srcByteData;
return PushWrite(&copysrcByteData, 1);
}

int CFifoBuffer::PushWrite(char unsigned *srcByteDataArray, int srcArraySize)
{
int i, WriteSize;

// extend buffer if allowed
if ((GetWritableBufferSize() < srcArraySize)&&
	((AutoExtend == 1)&&(AutoExtendSize > 0)))
	{
	SetBufferSize(BufferSize+(((srcArraySize-GetWritableBufferSize())/AutoExtendSize)*AutoExtendSize)
			+((((srcArraySize-GetWritableBufferSize())%AutoExtendSize) > 0)?AutoExtendSize:0));
	}

// buffer shift
WriteSize=min(srcArraySize, GetWritableBufferSize());
for(i=GetReadableBufferSize()-1;i>=0;i--)
	Buffer[(ReaderCursorPos+WriteSize+i)%BufferSize]=
		Buffer[(ReaderCursorPos+i)%BufferSize];

// buffer write
for(i=0;i<WriteSize;i++)
	Buffer[(ReaderCursorPos+i)%BufferSize]=srcByteDataArray[i];
WriterCursorPos+=WriteSize; if (WriterCursorPos >= BufferSize) WriterCursorPos=(WriterCursorPos%BufferSize);

return WriteSize;
}

#ifdef CCLASSDEBUG
void CFifoBuffer::Dump()
{
int i;
printf("BufferSize=%i AutoExtend=%i AutoExtendSize=%i ReaderCursorPos=%i WriterCursorPos=%i\n",
	BufferSize, AutoExtend, AutoExtendSize, ReaderCursorPos, WriterCursorPos);
printf("Buffer=["); for(i=0;i<BufferSize;i++) printf("%02X", Buffer[i]); printf("]\n");
}

void CFifoBuffer::SelfTest()
{
CFifoBuffer Buf;
char unsigned DummyData[] = "This is the self test.";
int i;
char unsigned LocalBuffer[8];

printf("Self testing CFifoBuffer() class..\n");

printf("  step 1: fixed buffer test.\n");
Buf.SetBufferSize(32);
Buf.SetAutoExtend(0, 0);
Buf.Empty();

printf("    Write [%s] ... returned %i\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData))); Buf.Dump();
printf("    Write [%s] ... returned %i (< %i)\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData)), strlen((const char *)DummyData)); Buf.Dump();
printf("    Read 6 bytes ... %i:\n", Buf.Read(LocalBuffer, 6));
printf("      [%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5]); Buf.Dump();
printf("    Write [ab] ... returned %i\n", Buf.Write((char unsigned *)"ab", 2)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [cde] ... returned %i\n", Buf.Write((char unsigned *)"cde", 3)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [fghi] ... returned %i\n", Buf.Write((char unsigned *)"fghi", 4)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [jklmn] ... returned %i\n", Buf.Write((char unsigned *)"jklmn", 5)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();

printf("  step 2: variable (auto extend) buffer test.\n");
Buf.SetBufferSize(32);
Buf.SetAutoExtend(1, 3);
Buf.Empty();

printf("    Write [%s] ... returned %i\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData))); Buf.Dump();
printf("    Write [%s] ... returned %i (== %i)\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData)), strlen((const char *)DummyData)); Buf.Dump();
printf("    Read 6 bytes ... %i:\n", Buf.Read(LocalBuffer, 6));
printf("      [%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5]); Buf.Dump();
printf("    Write [ab] ... returned %i\n", Buf.Write((char unsigned *)"ab", 2)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [cde] ... returned %i\n", Buf.Write((char unsigned *)"cde", 3)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [fghi] ... returned %i\n", Buf.Write((char unsigned *)"fghi", 4)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [jklmn] ... returned %i\n", Buf.Write((char unsigned *)"jklmn", 5)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();

printf("  step 3: buffer push (intrusion) test without auto extension.\n");
Buf.SetBufferSize(32);
Buf.SetAutoExtend(0, 0);
Buf.Empty();

printf("    Write [%s] ... returned %i\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData))); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Write [%s] ... returned %i\n", DummyData, Buf.Write(DummyData, strlen((const char *)DummyData))); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Push Write [@123] ... returned %i\n", Buf.PushWrite((char unsigned *)"@123", 4)); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();
printf("    Read 8 bytes ... %i:\n", Buf.Read(LocalBuffer, 8));
printf("      [%c%c%c%c%c%c%c%c]\n", LocalBuffer[0], LocalBuffer[1], LocalBuffer[2], LocalBuffer[3], LocalBuffer[4], LocalBuffer[5], LocalBuffer[6], LocalBuffer[7]); Buf.Dump();

printf("  step 4: buffer push (intrusion) test with auto extension.\n");
Buf.SetBufferSize(32);
Buf.SetAutoExtend(1, 3);
Buf.Empty();

printf("Self testing CFifoBuffer() completed.\n");
}
#endif
