#ifndef	CArray_defined
#define	CArray_defined

#include <malloc.h>
#include "CException.h"
#ifdef CCLASSDEBUG
#include <stdio.h>
#include "CString.h"
#endif

template<class TYPE, class ARG_TYPE=TYPE>
class CArray
{
public:
	// constructors
	CArray();
	CArray(CArray& sourceArray);

	// destructors
	~CArray();

	// methods
	void SetSize(int newSize, int newGrowByCount=-1);
	int IsEmpty();
	void SetAtGrow(int newGrowByCount);
	void RemoveAt(int iIndex, int iCount=1);
	void RemoveAll();

	TYPE& GetAt(int iIndex);
	void SetAt(int iIndex, ARG_TYPE newItem);
	TYPE **GetData();
	void Add(ARG_TYPE newItem);
	void Add(CArray& srcArray);
	void Insert(ARG_TYPE newItem, int iIndex);
	void Insert(CArray& srcArray, int iIndex);
	void Copy(const CArray& srcArray);

	// sorting
	void SwapItem(int Pos1, int Pos2);
	int QuickSort_Sub_Partition(int(*compar)(ARG_TYPE *, ARG_TYPE *), int First, int Last);
	void QuickSort_Sub(int(*compar)(ARG_TYPE *, ARG_TYPE *), int First, int Last);
	void QuickSort(int(*compar)(ARG_TYPE *, ARG_TYPE *));

	// operators
	CArray& operator=(const CArray& srcArray);
	TYPE& operator[](int iIndex);

	// self testing
	#ifdef CCLASSDEBUG
	void Dump();
	#endif

	int Size;
private:
	void ReallocData(int newArraySize);

	TYPE **ArrayData;
	int AllocatedSize;
	int GrowByCount;
};

// constructors
template<class TYPE, class ARG_TYPE>
CArray<TYPE, ARG_TYPE>::CArray()
{
ArrayData=NULL;
Size=0;
AllocatedSize=0;
GrowByCount=1;
}

// constructors
template<class TYPE, class ARG_TYPE>
CArray<TYPE, ARG_TYPE>::CArray(CArray& sourceArray)
{
int i;

ArrayData=NULL;
Size=0;
AllocatedSize=0;
GrowByCount=1;

SetSize(sourceArray.Size);
for(i=0;i<sourceArray.Size;i++) *ArrayData[i]=*sourceArray[i];
}

// destructors
template<class TYPE, class ARG_TYPE>
CArray<TYPE, ARG_TYPE>::~CArray()
{
if (IsEmpty() == 0) RemoveAll();
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::SetSize(int newSize, int newGrowByCount)
{
ReallocData(newSize);
if (newGrowByCount > 0) SetAtGrow(newGrowByCount);
}

// methods
template<class TYPE, class ARG_TYPE>
int CArray<TYPE, ARG_TYPE>::IsEmpty()
{
if (Size < 1) return 1;
else return 0;
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::SetAtGrow(int newGrowByCount)
{
if (newGrowByCount > 0)
	GrowByCount=newGrowByCount;
else	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::SetAtGrow"));
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::RemoveAt(int iIndex, int iCount)
{
int i;

if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::RemoveAt"));
if ((iCount < 1)||(iIndex+iCount >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::RemoveAt"));

for(i=iIndex+iCount;i<Size;i++) *ArrayData[i-iCount]=*ArrayData[i];
ReallocData(Size-iCount);
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::RemoveAll()
{
if (ArrayData != NULL) free(ArrayData);
ArrayData=NULL;
Size=0;
AllocatedSize=0;
}

// methods
template<class TYPE, class ARG_TYPE>
TYPE& CArray<TYPE, ARG_TYPE>::GetAt(int iIndex)
{
if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::GetAt"));

return *ArrayData[iIndex];
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::SetAt(int iIndex, ARG_TYPE newItem)
{
if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::GetAt"));

*ArrayData[iIndex]=newItem;
}

template<class TYPE, class ARG_TYPE>
TYPE **CArray<TYPE, ARG_TYPE>::GetData()
{
// only returns bulk data
return ArrayData;
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Add(ARG_TYPE newItem)
{
int Pos;

Pos=Size;
ReallocData(Size+1);
*ArrayData[Pos]=newItem;
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Add(CArray& srcArray)
{
int Pos, i;
TYPE **srcArrayData;

Pos=Size;
ReallocData(Size+srcArray.Size);
srcArrayData=srcArray.GetData();
for(i=0;i<srcArray.Size;i++) *ArrayData[Pos+i]=*srcArrayData[i];
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Insert(ARG_TYPE newItem, int iIndex)
{
int Pos, i;

if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::Insert"));

Pos=Size;
ReallocData(Size+1);
for(i=Pos-1;i>=iIndex;i--) *ArrayData[i+1]=*ArrayData[i];
*ArrayData[iIndex]=newItem;
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Insert(CArray& srcArray, int iIndex)
{
int Pos, i;
TYPE **srcArrayData;

if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::Insert"));

Pos=Size;
ReallocData(Size+srcArray.Size);
srcArrayData=srcArray.GetData();
for(i=Pos-1;i>=iIndex;i--) *ArrayData[i+srcArray.Size]=*ArrayData[i];
for(i=0;i<srcArray.Size;i++) *ArrayData[iIndex+i]=*srcArrayData[i];
}

// methods
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Copy(const CArray& srcArray)
{
int i;
TYPE **srcArrayData;

RemoveAll();
ReallocData(srcArray.Size);
srcArrayData=((CArray&)srcArray).GetData();
for(i=0;i<srcArray.Size;i++) *ArrayData[i]=*srcArrayData[i];
}


// sorting
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::SwapItem(int Pos1, int Pos2)
{
ARG_TYPE *TmpItem;

TmpItem=ArrayData[Pos2];
ArrayData[Pos2]=ArrayData[Pos1];
ArrayData[Pos1]=TmpItem;
}

// sorting
template<class TYPE, class ARG_TYPE>
int CArray<TYPE, ARG_TYPE>::QuickSort_Sub_Partition(int(*compar)(ARG_TYPE *, ARG_TYPE *), int First, int Last)
{
int i, l;

SwapItem(First, (First+Last)/2);
l=First;
for(i=First+1;i<=Last;i++)
	{
	if ((*compar)(ArrayData[i], ArrayData[First]) < 0)
		{
		l++;
		SwapItem(l, i);
		}
	}
SwapItem(First, l);

return l;
}

// sorting
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::QuickSort_Sub(int(*compar)(ARG_TYPE *, ARG_TYPE *), int First, int Last)
{
if (First < Last)
	{
	int p;
	
	p=QuickSort_Sub_Partition(compar, First, Last);
	QuickSort_Sub(compar, p+1, Last);
	QuickSort_Sub(compar, First, p-1);
	}
}

// sorting
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::QuickSort(int(*compar)(ARG_TYPE *, ARG_TYPE *))
{
if (Size < 1) return;
QuickSort_Sub(compar, 0, Size-1);
}

// operators
template<class TYPE, class ARG_TYPE>
CArray<TYPE, ARG_TYPE>& CArray<TYPE, ARG_TYPE>::operator=(const CArray& srcArray)
{
this->Copy(srcArray);
return (*this);
}

// operators
template<class TYPE, class ARG_TYPE>
TYPE& CArray<TYPE, ARG_TYPE>::operator[](int iIndex)
{
if ((iIndex < 0)||(iIndex >= Size))
	throw(CException(CException::CErrorTypeException,
		CException::CIllegalParameter, "CArray::operator[]"));

return *ArrayData[iIndex];
}

template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::ReallocData(int newArraySize)
{
int newArraySizeAligned;
int i;

newArraySizeAligned=(newArraySize/GrowByCount)*GrowByCount
	+(((newArraySize%GrowByCount) > 0)?1:0);

if (Size == newArraySize) return;	// do nothing
if (newArraySizeAligned < 1)
	{
	// delete all array items
	for(i=0;i<Size;i++) delete ArrayData[i];
	delete[] (char unsigned *)ArrayData;
	ArrayData=NULL;
	AllocatedSize=0;
	Size=0;
	}
else if (Size < 1)
	{
	// create new array
	ArrayData=(TYPE **)new char unsigned[(size_t)newArraySizeAligned*sizeof(TYPE *)];
	memset((void *)ArrayData, 0, (size_t)newArraySizeAligned*sizeof(TYPE *));
	for(i=0;i<newArraySize;i++) ArrayData[i]=new TYPE();
	AllocatedSize=newArraySizeAligned;
	Size=newArraySize;
	}
else if (Size > newArraySize)
	{
	// shrink array
	for(i=newArraySize;i<Size;i++) delete ArrayData[i];

	if (AllocatedSize < newArraySizeAligned)
		throw(CException(CException::CErrorTypeException,
			CException::CInternalError, "CArray::ReallocData"));
	else if (AllocatedSize > newArraySizeAligned)
		{
		TYPE **newArrayData;

		// reallocate
		newArrayData=(TYPE **)new char unsigned[(size_t)newArraySizeAligned*sizeof(TYPE *)];
		memcpy((void *)newArrayData, (void *)ArrayData, (size_t)newArraySize*sizeof(TYPE *));
		delete[] (char unsigned *)ArrayData;
		ArrayData=newArrayData;
		}

	AllocatedSize=newArraySizeAligned;
	Size=newArraySize;
	}
else	{
	// expand array
	if (AllocatedSize > newArraySizeAligned)
		throw(CException(CException::CErrorTypeException,
			CException::CInternalError, "CArray::ReallocData"));
	else if (AllocatedSize < newArraySizeAligned)
		{
		TYPE **newArrayData;

		// reallocate
		newArrayData=(TYPE **)new char unsigned[(size_t)newArraySizeAligned*sizeof(TYPE *)];
		memcpy((void *)newArrayData, (void *)ArrayData, (size_t)Size*sizeof(TYPE *));
		delete[] (char unsigned *)ArrayData;
		ArrayData=newArrayData;
		}

	for(i=Size;i<newArraySize;i++) ArrayData[i]=new TYPE();

	AllocatedSize=newArraySizeAligned;
	Size=newArraySize;
	}

return;
}

#ifdef CCLASSDEBUG
template<class TYPE, class ARG_TYPE>
void CArray<TYPE, ARG_TYPE>::Dump()
{
int i;

printf("[");
for(i=0;i<Size;i++) printf("%s, ", (const char *)(*ArrayData[i]));
printf("]");
}
#endif

// self testing
#ifdef CCLASSDEBUG
void CArray_SelfTest();
#endif

#endif
